<?php
class WC_Gateway_Shepa extends WC_Payment_Gateway
{
    /**
     * Whether the sndbox mode is enabled
     *
     * @var bool
     */
    public $sandbox;

    /**
     * Api key
     *
     * @var string
     */
    public $api;

    public function __construct()
    {
        $this->id = "shepa";
        $this->has_fields = false;
        $logo_path = '/assets/images/logo_vm.png';
        $this->icon = WC_Shepa_Dollar_Payments::plugin_url() . $logo_path;
        $this->method_title = __('Visa&Mastercard', 'wc-gateway-shepa');
        $this->method_description = __("Shepa payment gateway for international payment", "translate");
        $this->supports = [
            'products',
            'refunds'
        ];

        //Load the settings
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables
        $this->title = $this->get_option("title");
        $this->description = $this->get_option("description");
        $this->sandbox = 'yes' === $this->get_option("sandbox", 'no');
        // $this->sandbox = $this->get_option("sandbox");
        $this->api = $this->get_option("api");

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [
            $this,
            'process_admin_options'
        ]);
        add_action('woocommerce_receipt_' . $this->id, [
            $this,
            'process_payment_request'
        ]);
        // add_action("woocommerce_api_callback" , '')
        add_action('woocommerce_api_' . strtolower(get_class($this)) . '', array($this, 'process_payment_verify'));
    }

    public function get_icon_url()
    {
        return $this->icon;
    }

    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled' => [
                'title' => __("Enabled", 'translate'),
                'type' => 'checkbox',
                'label' => __("Enable Shepa Payment", 'translate'),
                'default' => 'no'
            ],
            'sandbox' => [
                'title' => __("Sandbox Mode", 'translate'),
                'type' => "checkbox",
                'label' => __("Enable Shepa Sandbox"),
                'default' => 'no',
                'description' => __('Shepa sandbox can be used to test payments.', 'translate'),
            ],
            'api' => [
                'title' => __('API key', 'translate'),
                'type' => 'text',
                'default' => '',
                'desc_tip' => true,
                'description' => __('To use the "Sandbox Mode" set this field to sandbox.', 'translate'),
            ],
            'title'                 => [
                'title'       => __('Title', 'translate'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'woocommerce'),
                'default'     => __('Debit/Creadit card', 'translate'),
                'desc_tip'    => true,
            ],
            'description'           => [
                'title'       => __('Description', 'translate'),
                'type'        => 'text',
                'desc_tip'    => true,
                'description' => __('This controls the description which the user sees during checkout.', 'woocommerce'),
                'default'     => __("checkout by Debit/Creadit card", 'translate'),
            ],
        ];
    }

    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        return [
            'result' => 'success',
            'redirect' => $order->get_checkout_payment_url(true)
        ];
    }
    public function process_payment_request($order_id)
    {
        $order = wc_get_order($order_id);
        $amount = $order->get_total();
        $callback = add_query_arg('wc_order', $order_id, WC()->api_request_url(__CLASS__));
        $orderObject = [];
        $orderObject['total'] = $amount;
        $orderObject['billing'] = $this->get_address($order);
        ($order->has_shipping_address()) ?  $orderObject['shipping'] = $this->get_address($order, false) : null;
        $orderObject['products'] = $this->get_products_list($order);

        for ($i = 0; $i < count($orderObject['products']); $i++) {
            $image = wp_get_attachment_image_url(wc_get_product($orderObject['products'][0]['id'])->get_image_id());
            $orderObject['products'][$i]['image'] = $image;
        }
        // call to shepa
        $url = "https://merchant.shepa.com/api/v1/token";
        if ($this->sandbox) {
            $url = "https://sandbox.shepa.com/api/v1/token";
        }
        $params = [
            'api' => $this->api,
            'amount' => $amount,
            'callback' => $callback,
            'orderId' => $order_id,
            'order' => $orderObject
        ];

        $output = $this->curl_post($url, $params);
        $Message = "";
        if ($output->success == true) {
            wp_redirect($output->result->url);
            exit;
        } else {
            $Message = ' تراکنش ناموفق بود <br>';
            if (!empty($output->errors)) $Message .= implode("<br>", $output->errors);
        }
        if (!empty($Message)) {
            //!!change
            $Note = sprintf(__('خطا در هنگام ارسال به بانک : %s', 'woocommerce'), $Message);

            $order->set_status('failed', $Message);

            $Notice = sprintf(__('در هنگام اتصال به بانک خطای زیر رخ داده است : <br/>%s', 'woocommerce'), $Message);
            wc_add_notice($Notice, 'error');
        }
    }

    public function process_payment_verify()
    {
        global $woocommerce;
        $status = $_GET['status'] ?? '';
        $order = wc_get_order($_GET['wc_order']);
        if (!$order->needs_payment()) {
            wp_redirect(add_query_arg('wc_status', 'success', $this->get_return_url($order)));
            exit;
        }
        if ($status == 'failed') {
            $order->set_status('failed', "Payment failed");
            wc_add_notice("Payment failed", 'error');
            wp_redirect($woocommerce->cart->get_checkout_url());
            exit;
        }
        $token = $_GET["token"];
        $amount = $order->get_total();
        $data = [
            'amount' => $amount,
            'token' => $token,
            'api' => $this->api,
            'orderId' => $order->get_id()
        ];
        $url = "https://merchant.shepa.com/api/v1/verify";
        if ($this->sandbox) {
            $url = "https://sandbox.shepa.com/api/v1/verify";
        }
        $response = $this->curl_post($url, $data);
        if ($response->success == true) {
            $order->payment_complete($response->result->transaction_id);
            $order->set_meta_data([
                'ref_id' => $response->result->refid,
                'date' => $response->result->date
            ]);
            $note = "refid: %s";
            $order->add_order_note(sprintf($note, $response->result->refid));
            $woocommerce->cart->empty_cart();
            wp_redirect(add_query_arg('wc_status', 'success', $this->get_return_url($order)));
            exit;
        } else {
            $order->set_status('failed', "Payment failed");
            $errors = "";
            foreach ($response->errors as $error) {
                $errors .= $error . PHP_EOL;
            }
            $order->add_order_note("ERROR: " . $errors);
            wc_add_notice("ERROR: " . $errors, 'error');
            wp_redirect($woocommerce->cart->get_checkout_url());
            exit;
        }
    }

    private function curl_post($url, $data)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
        ]);
        $response = curl_exec($ch);
        curl_close($ch);
        return json_decode($response);
    }

    /**
     * create an array of address
     *
     * @param \WC_Order $order
     * @return array
     */
    private function get_products_list($order)
    {
        $items = $order->get_items();
        $products = [];
        foreach ($items as $item) {
            $product = [
                "id" => $item['product_id'],
                "name" => $item['name'],
                "price" => $item['subtotal'],
                "qty" => $item['quantity']
            ];
            $products[] = $product;
        }
        return $products;
    }

    /**
     * create an array of address
     *
     * @param \WC_Order $order
     * @param boolean $billing
     * @return array
     */
    private function get_address($order, $billing = true)
    {
        $result = [];
        if ($billing) {
            $result['first_name'] = $order->get_billing_first_name();
            $result['last_name'] = $order->get_billing_last_name();
            $result['address_1'] = $order->get_billing_address_1();
            $result['city'] = $order->get_billing_city();
            $result['state'] = $order->get_billing_state();
            $result['postcode'] = $order->get_billing_postcode();
            $result['country'] = $order->get_billing_country();
            $result['email'] = $order->get_billing_email();
            $result['phone'] = $order->get_billing_phone();
        } else {
            $result['first_name'] = $order->get_shipping_first_name();
            $result['last_name'] = $order->get_shipping_last_name();
            $result['address_1'] = $order->get_shipping_address_1();
            $result['city'] = $order->get_shipping_city();
            $result['state'] = $order->get_shipping_state();
            $result['postcode'] = $order->get_shipping_postcode();
            $result['country'] = $order->get_shipping_country();
            $result['email'] = $order->get_billing_email();
            $result['phone'] = $order->get_shipping_phone();
        }
        return $result;
    }
}
